/*
 *  Native support for the I/O-Warrior
 *	for kernel 2.4.x only
 *
 *  Copyright (c) 2003 Code Mercenaries GmbH
 *  written by Christian Lucht <lucht@codemercs.com>
 *
 *  based on
 *  usb-skeleton.c by Greg Kroah-Hartman  <greg@kroah.com>
 *  brlvger.c by Stephane Dalton  <sdalton@videotron.ca>
 *           and Stphane Doyon   <s.doyon@videotron.ca>
 *
 *  ChangeLog:
 *      v0.1 	- Initial release, simple I/O only
 *      v0.2 	- access to special function endpoints implemented
 *		v0.2.1	- hot plug bug fixed
 */

/*
 * compile-command:
 *     "gcc -D__KERNEL__ -I/usr/src/linux/include -Wall -Wstrict-prototypes -O2 -fomit-frame-pointer -pipe -fno-strength-reduce -mcpu=i486 -DMODULE -DMODVERSIONS -include /usr/src/linux/include/linux/modversions.h  -c -o iowarrior.o iowarrior.c"
 *
 * for kernel version 2.4.26 or later its:
 *     "gcc -D__KERNEL__ -I/lib/modules/`uname -r`/build/include -Wall -Wstrict-prototypes -O2 -fomit-frame-pointer -pipe -fno-strength-reduce -mcpu=i486 -DMODULE -DMODVERSIONS -include /lib/modules/`uname -r`/build/include/linux/modversions.h  -c -o iowarrior.o iowarrior.c"
 */


#include <linux/module.h>
#include <linux/usb.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/poll.h>
#include <linux/devfs_fs_kernel.h>
#include "iowarrior.h"

/* Use our own dbg macro */
#undef dbg
#define dbg( format, arg... ) do { if( debug ) printk( KERN_DEBUG __FILE__ ": " format "\n" , ## arg ); } while ( 0 )


MODULE_AUTHOR( DRIVER_AUTHOR );
MODULE_DESCRIPTION( DRIVER_DESC );
MODULE_LICENSE( "GPL" );

/* Module paramaters */
static int debug = 0;
MODULE_PARM( debug, "i" );
MODULE_PARM_DESC( debug, "Debug enabled or not" );


/*---------------------------*/
/* local function prototypes */
/*---------------------------*/
static void * iowarrior_probe( struct usb_device *udev, unsigned int ifnum, const struct usb_device_id *id );
static void iowarrior_disconnect( struct usb_device *dev, void *ptr );

static ssize_t iowarrior_read( struct file *file, char *buffer, size_t count, loff_t *ppos );
static int iowarrior_ioctl( struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg );
static int iowarrior_open( struct inode *inode, struct file *file );
static int iowarrior_release( struct inode *inode, struct file *file );
static unsigned iowarrior_poll( struct file *file, poll_table *wait );

static void iowarrior_callback( struct urb *urb );

/*--------------*/
/*     data     */
/*--------------*/

/* Structure to hold all of our device specific stuff */
struct iowarrior {
	struct usb_device *		udev;				/* save off the usb device pointer */
	int						ifnum;				/* number of the interface */
	devfs_handle_t			devfs;				/* devfs device node */
	int						minor;				/* the starting minor number for this device */

	unsigned char *			ctrl_out_buffer;	/* the buffer to send data */
	__u8					ctrl_out_endpointAddr;	/* the address of the interrupt out endpoint */

	struct urb *			int_in_urb;			/* the urb used to transmit read data */
	unsigned char *			int_in_buffer;		/* the buffer to read data */
	unsigned char			serial_number;		/* to detect lost packages */
	__u8					int_in_endpointAddr;	/* the address of the interrupt in endpoint */
	__u8					int_in_interval;	/* the interrupt interval value */

	unsigned int			packet_size;		/* depends on IOWarrior type and interface */
	unsigned char *			read_queue;			/* size is MAX_INTERRUPT_BUFFER * packet size */
	wait_queue_head_t		read_wait;
	atomic_t				read_idx;
	atomic_t				intr_idx;
	spinlock_t				intr_idx_lock;		/* protects intr_idx */
	atomic_t				overflow_flag;		/* signals an index 'rollover' */

	int						present;				/* if the device is not disconnected */
	int						opened;
	struct semaphore		sem;				/* locks this structure */
	struct semaphore		open_sem;			/* protects ->opened */
};


/*--------------*/
/*    globals   */
/*--------------*/
/*  the global usb devfs handle */
extern devfs_handle_t usb_devfs_handle;			/* /dev/usb dir */

/* array of pointers to our devices that are currently connected */
static struct iowarrior *minor_table[MAX_DEVICES];

/* lock to protect the minor_table structure */
static DECLARE_MUTEX( minor_table_mutex );


/*---------------------*/
/* driver registration */
/*---------------------*/
/* table of devices that work with this driver */
struct usb_device_id iowarrior_ids[] = {
	{ USB_DEVICE(USB_VENDOR_ID_CODEMERCS, USB_DEVICE_ID_CODEMERCS_IOW40) },
	{ USB_DEVICE(USB_VENDOR_ID_CODEMERCS, USB_DEVICE_ID_CODEMERCS_IOW24) },
	{}				/* Terminating entry */
};
MODULE_DEVICE_TABLE( usb, iowarrior_ids );


/*
 * File operations needed when we register this driver.
 * This assumes that this driver NEEDS file operations,
 * of course, which means that the driver is expected
 * to have a node in the /dev directory. If the USB
 * device were for a network interface then the driver
 * would use "struct net_driver" instead, and a serial
 * device would use "struct tty_driver".
 */
static struct file_operations iowarrior_fops = {
	/*
	 * The owner field is part of the module-locking
	 * mechanism. The idea is that the kernel knows
	 * which module to increment the use-counter of
	 * BEFORE it calls the device's open() function.
	 * This also means that the kernel can decrement
	 * the use-counter again before calling release()
	 * or should the open() function fail.
	 *
	 * Not all device structures have an "owner" field
	 * yet. "struct file_operations" and "struct net_device"
	 * do, while "struct tty_driver" does not. If the struct
	 * has an "owner" field, then initialize it to the value
	 * THIS_MODULE and the kernel will handle all module
	 * locking for you automatically. Otherwise, you must
	 * increment the use-counter in the open() function
	 * and decrement it again in the release() function
	 * yourself.
	 */
	owner:		THIS_MODULE,

	read:		iowarrior_read,
	ioctl:		iowarrior_ioctl,
	open:		iowarrior_open,
	release:	iowarrior_release,
	poll:		iowarrior_poll
};


/* usb specific object needed to register this driver with the usb subsystem */
static struct usb_driver iowarrior_driver = {
	name:		"iowarrior",
	probe:		iowarrior_probe,
	disconnect:	iowarrior_disconnect,
	fops:		&iowarrior_fops,
	minor:		IOWARRIOR_MINOR_BASE,
	id_table:	iowarrior_ids
};


/**
 *	iowarrior_init
 */
static int __init iowarrior_init( void )
{
	int result;

	memset( minor_table, 0, sizeof( minor_table ));

	/* register this driver with the USB subsystem */
	result = usb_register( &iowarrior_driver );
	if( result < 0 )
	{
		err( "usb_register failed for the "__FILE__" driver. Error number %d", result );
		return -1;
	}

	info( DRIVER_DESC " " DRIVER_VERSION );
	return 0;
}


/**
 *	iowarrior_exit
 */
static void __exit iowarrior_exit( void )
{
	/* deregister this driver with the USB subsystem */
	usb_deregister( &iowarrior_driver );
}


module_init( iowarrior_init );
module_exit( iowarrior_exit );


/**
 *	iowarrior_delete
 */
static inline void iowarrior_delete( struct iowarrior *dev )
{
	minor_table[dev->minor] = NULL;

	if( dev->ctrl_out_buffer != NULL )
		kfree( dev->ctrl_out_buffer );

	if(dev->int_in_buffer != NULL )
		kfree( dev->int_in_buffer );

	if( dev->int_in_urb != NULL )
		usb_free_urb( dev->int_in_urb );

	if( dev->read_queue != NULL )
		kfree( dev->read_queue );

	kfree( dev );
}


/*---------------------------------*/
/*  probe and disconnect functions */
/*---------------------------------*/
/**
 *	iowarrior_probe
 *
 *	Called by the usb core when a new device is connected that it thinks
 *	this driver might be interested in.
 */
static void * iowarrior_probe( struct usb_device *udev, unsigned int ifnum, const struct usb_device_id *id )
{
	struct iowarrior *dev = NULL;
	struct usb_interface *interface;
	struct usb_interface_descriptor *iface_desc;
	struct usb_endpoint_descriptor *endpoint;
	int minor;
	int buffer_size;

	/* See if the device offered us matches what we can accept */
	if(( udev->descriptor.idVendor != USB_VENDOR_ID_CODEMERCS ) ||
	  (( udev->descriptor.idProduct != USB_DEVICE_ID_CODEMERCS_IOW40 ) &&
	   ( udev->descriptor.idProduct != USB_DEVICE_ID_CODEMERCS_IOW24 )))
	{
		return NULL;
	}

	/* select a "subminor" number (part of a minor number) */
	down( &minor_table_mutex );
	for( minor = 0; minor<MAX_DEVICES; ++minor )
	{
		if( minor_table[minor] == NULL )
			break;
	}
	if( minor >= MAX_DEVICES )
	{
		info( "Too many devices plugged in, can not handle this device." );
		goto exit;
	}

	/* allocate memory for our device state and intialize it */
	dev = kmalloc( sizeof( struct iowarrior ), GFP_KERNEL );
	if( dev == NULL )
	{
		err( "Out of memory" );
		goto exit;
	}
	memset( dev, 0x00, sizeof( *dev ));
	atomic_set( &dev->intr_idx, 0 );
	atomic_set( &dev->read_idx, 0 );
	spin_lock_init( &dev->intr_idx_lock );
	atomic_set( &dev->overflow_flag, 0 );
	init_waitqueue_head( &dev->read_wait );
	/* opened is memset'ed to 0 */
	init_MUTEX( &dev->open_sem );

	interface = &udev->actconfig->interface[ifnum];

	init_MUTEX( &dev->sem );
	dev->udev = udev;
	dev->ifnum = ifnum;
	dev->minor = minor;

	/* set up the endpoint information */
	/* check out the endpoints */
	iface_desc = &interface->altsetting[0];

	/* there is only one endpoint on every interface */
	endpoint = &iface_desc->endpoint[0];
	if( !( endpoint->bEndpointAddress & USB_DIR_IN ) ||
	  (( endpoint->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK ) != USB_ENDPOINT_XFER_INT ))
	{
		err ( "wrong endpoint" );
		goto error;
	}

	/* we found the interrupt in endpoint */
	dev->packet_size = endpoint->wMaxPacketSize;
	buffer_size = dev->packet_size;
	dev->int_in_endpointAddr = endpoint->bEndpointAddress;

	dev->int_in_buffer = kmalloc( buffer_size, GFP_KERNEL );
	if( !dev->int_in_buffer )
	{
		err( "Couldn't allocate int_in_buffer" );
		goto error;
	}

	/* +1 is for the time stamp */
	dev->read_queue = kmalloc((( buffer_size + 1 ) * MAX_INTERRUPT_BUFFER ), GFP_KERNEL );
	if( !dev->read_queue )
	{
		err( "Couldn't allocate read_queue" );
		goto error;
	}

	/* allow device read and ioctl */
	dev->present = 1;

	dev->int_in_interval = endpoint->bInterval;

	/* initialize the devfs node for this device and register it */
	char name[15];
	sprintf( name, "iowarrior%d", dev->minor );

	dev->devfs = devfs_register ( usb_devfs_handle, name,
	                              DEVFS_FL_DEFAULT, USB_MAJOR,
	                              IOWARRIOR_MINOR_BASE + dev->minor,
	                              S_IFCHR | S_IRUSR | S_IWUSR |
	                              S_IRGRP | S_IWGRP | S_IROTH,
	                              &iowarrior_fops, NULL );

  	minor_table[minor] = dev;

	/* let the user know what node this device is now attached to */
	info( "I/O-Warrior device now attached to iowarrior%d", dev->minor );

	goto exit;

error:
	iowarrior_delete( dev );
	dev = NULL;

exit:
	up( &minor_table_mutex );
	return dev;
}


/**
 *	iowarrior_disconnect
 *
 *	Called by the usb core when the device is removed from the system.
 */
static void iowarrior_disconnect( struct usb_device *udev, void *ptr )
{
	struct iowarrior *dev;
	int minor;

	dev = (struct iowarrior *)ptr;

	down( &minor_table_mutex );
	down( &dev->sem );

	minor = dev->minor;

	/* remove our devfs node */
	devfs_unregister( dev->devfs );

	/* prevent device read, write and ioctl */
	dev->present = 0;

	/* if the device is not opened, then we clean up right now */
	if( !dev->opened )
	{
		up( &dev->sem );
		iowarrior_delete( dev );
	}
	else
	{
		dev->udev = NULL;
		wake_up_interruptible( &dev->read_wait );
		up( &dev->sem );
	}

	info( "I/O-Warror #%d now disconnected", minor );
	up( &minor_table_mutex );
}


/*---------------------*/
/* fops implementation */
/*---------------------*/

static int read_index( struct iowarrior *dev )
{
	int intr_idx, read_idx;

	read_idx = atomic_read( &dev->read_idx );
	intr_idx = atomic_read( &dev->intr_idx );

	return( read_idx == intr_idx ? -1 : read_idx );
}


 /**
  *  iowarrior_read
  */
static ssize_t iowarrior_read( struct file *file, char *buffer, size_t count, loff_t *ppos )
{
	struct iowarrior *dev;
	int read_idx;

	dev = (struct iowarrior *)file->private_data;

	dbg( "%s - minor %d, count = %d", __func__, dev->minor, count );

	/* read count must be packet size (+ time stamp) */
	if(( count !=  dev->packet_size ) && ( count != ( dev->packet_size + 1 )))
		return -EINVAL;

	/* verify that the device wasn't unplugged */
	if( dev->udev == NULL )
		return -ENODEV;

	/* repeat until no buffer overrun in callback handler occur */
	do
	{
		atomic_set( &dev->overflow_flag, 0 );
		if(( read_idx = read_index( dev )) == -1 )
		{
			/* queue emty */
			if( file->f_flags & O_NONBLOCK )
				return -EAGAIN;
			else
			{
				int r = wait_event_interruptible( dev->read_wait,
				                                 ( !dev->udev || ( read_idx = read_index( dev )) != -1 ));
				if( !dev->udev )
					return -ENOLINK;
				if( r )
					return r;
				if( read_idx == -1 )
					/* should not happen */
					return 0;
			}
		}

		int offset = read_idx * (dev->packet_size + 1 );
		if( copy_to_user( buffer, dev->read_queue+offset, ( dev->packet_size + 1 )))
			return -EFAULT;
	}
	while( atomic_read( &dev->overflow_flag ));

	read_idx = ++read_idx == MAX_INTERRUPT_BUFFER ? 0 : read_idx;
	atomic_set( &dev->read_idx, read_idx );

	return count;
}


/**
 *	iowarrior_ioctl
 */
static int iowarrior_ioctl( struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg )
{
	struct iowarrior *dev;
	dev = (struct iowarrior *)file->private_data;
	__u8 buffer[8];		/* max packet size for iowarrior */

	if( dev == NULL )
	{
		return -ENODEV;
	}

	/* lock this object */
	down( &dev->sem );

	/* verify that the device wasn't unplugged */
//	if( dev->udev == NULL )
	if( !dev->present )
	{
		up( &dev->sem );
		return -ENODEV;
	}

	dbg( "%s - minor %d, cmd 0x%.4x, arg %ld", __func__, dev->minor, cmd, arg );

	int retval = 0;
	switch( cmd )
	{
		case IOW_WRITE:
		{
			int wrote;
			copy_from_user( (void *)buffer, (const void *)arg, dev->packet_size );

			// 2 = Report Type Output, 0 = Report IDs are not used
			wrote = usb_set_report( dev->udev, dev->ifnum, 2, 0, (void *)buffer, dev->packet_size );
			if( wrote < 0 )
				retval = wrote;

			break;
		}
		case IOW_READ:
		{
			int read;

			// 1 = Report Type Input, 0 = Report IDs are not used
			read = usb_get_report( dev->udev, dev->ifnum, 1, 0, (void *)buffer, dev->packet_size );
			if( read < 0 )
				retval = read;
			else
				copy_to_user( (void *)arg, (void *)buffer, dev->packet_size );
			break;
		}
		default:
		{
			/* return that we did not understand this ioctl call */
			retval = -ENOTTY;
			break;
		}
	}

	/* unlock the device */
	up( &dev->sem );
	return retval;
}


/**
 *	iowarrior_open
 */
static int iowarrior_open( struct inode *inode, struct file *file )
{
	struct iowarrior *dev = NULL;
	int subminor;
	int retval = 0;

	dbg( "%s", __func__ );

	subminor = MINOR( inode->i_rdev ) - IOWARRIOR_MINOR_BASE;
	if(( subminor < 0 ) ||
	  ( subminor >= MAX_DEVICES ))
	{
		return -ENODEV;
	}

	/* Increment our usage count for the module.
	 * This is redundant here, because "struct file_operations"
	 * has an "owner" field. This line is included here soley as
	 * a reference for drivers using lesser structures... ;-)
	 */
	MOD_INC_USE_COUNT;

	/* lock our minor table and get our local data for this minor */
	down( &minor_table_mutex );
	dev = minor_table[subminor];
	if( dev == NULL )
	{
		up( &minor_table_mutex );
		MOD_DEC_USE_COUNT;
		return -ENODEV;
	}

	/* lock this device */
	down( &dev->sem );

	/* unlock the minor table */
	up( &minor_table_mutex );

	/* Only one process can open each device, no sharing. */
	retval = -EBUSY;
	if( dev->opened )
		goto error;

	dbg( "Opening iowarrior%d", dev->minor );

	/* setup interrupt handler for receiving values */
	dev->int_in_urb = usb_alloc_urb( 0 );
	if( !dev->int_in_urb )
	{
		err( "Unable to allocate URB" );
		goto error;
	}

	FILL_INT_URB( dev->int_in_urb, dev->udev,
	              usb_rcvintpipe( dev->udev, dev->int_in_endpointAddr ),
	              dev->int_in_buffer, dev->packet_size,
	              iowarrior_callback, dev, dev->int_in_interval );

	if(( retval = usb_submit_urb( dev->int_in_urb)) < 0 )
	{
		err( "Error %d while submitting URB", retval );
		goto error;
	}

	/* increment our usage count for the driver */
	++dev->opened;
	/* save our object in the file's private structure */
	file->private_data = dev;
	retval = 0;
	goto out;

error:
	MOD_DEC_USE_COUNT;
out:
	/* unlock this device */
	up( &dev->sem );
	return retval;
}


/**
 *	iowarrior_release
 */
static int iowarrior_release (struct inode *inode, struct file *file)
{
	struct iowarrior *dev;
	int retval = 0;

	dev = (struct iowarrior *)file->private_data;
	if( dev == NULL )
	{
		dbg( "%s - object is NULL", __func__ );
		return -ENODEV;
	}

	dbg( "%s - minor %d", __func__, dev->minor );

	/* lock our minor table */
	down( &minor_table_mutex );

	/* lock our device */
	down( &dev->sem );

	if( dev->opened <= 0 )
	{
		dbg( "%s - device not opened", __func__ );
		retval = -ENODEV;
		goto exit_not_opened;
	}


	/* shutdown any reads that might be going on */
	usb_unlink_urb( dev->int_in_urb );

	--dev->opened;

	/* decrement our usage count for the module */
	MOD_DEC_USE_COUNT;

	if( !dev->present && !dev->opened )
	{
		/* the device was unplugged before the file was released */
		up( &dev->sem );
		up( &minor_table_mutex );
		iowarrior_delete( dev );
		return 0;
	}


exit_not_opened:
	up( &dev->sem );
	up( &minor_table_mutex );

	return retval;
}


static unsigned iowarrior_poll( struct file *file, poll_table *wait )
{
	struct iowarrior *dev = file->private_data;

	if( !dev->udev )
		return POLLERR | POLLHUP;

	poll_wait( file, &dev->read_wait, wait );

	if( !dev->udev )
		return POLLERR | POLLHUP;

	if( read_index( dev ) != -1 )
		return POLLIN | POLLRDNORM;

	return 0;
}


/*
 * USB callback handler for reading data
 */
static void iowarrior_callback (struct urb *urb)
{
	struct iowarrior *dev = (struct iowarrior *)urb->context;
	int intr_idx;
	int read_idx;
	int aux_idx;
	int offset;

	if( urb->status )
	{
		if( urb->status == -ETIMEDOUT )
			dbg( "Status -ETIMEDOUT, probably disconnected" );
		else if( urb->status != -ENOENT )
			err( "Status %d", urb->status );
		return;
	}

	spin_lock( &dev->intr_idx_lock );
	intr_idx = atomic_read( &dev->intr_idx );
	/* aux_idx become previous intr_idx */
	aux_idx = ( intr_idx == 0 ) ? ( MAX_INTERRUPT_BUFFER - 1 ) : ( intr_idx - 1 );
	read_idx = atomic_read( &dev->read_idx );

	/* queue is not empty and it's interface 0 */
	if(( intr_idx != read_idx ) && ( !dev->ifnum ))
	{
		/* + 1 for serial number */
		offset = aux_idx * ( dev->packet_size + 1 );
		if( !memcmp( dev->read_queue+offset, urb->transfer_buffer, dev->packet_size ));
		{
			/* equal values on interface 0 will be ignored */
			spin_unlock( &dev->intr_idx_lock );
			return;
		}
	}

	/* aux_idx become next intr_idx */
	aux_idx = ( intr_idx == ( MAX_INTERRUPT_BUFFER - 1 )) ? 0 : ( intr_idx + 1 );
	if( read_idx == aux_idx )
	{
		/* queue full, dropping oldest input */
		read_idx = ( ++read_idx == MAX_INTERRUPT_BUFFER ) ? 0 : read_idx;
		atomic_set( &dev->read_idx, read_idx );
		atomic_set( &dev->overflow_flag, 1 );
	}

	/* +1 for serial number */
	offset = intr_idx * ( dev->packet_size + 1 );
	memcpy( dev->read_queue+offset, urb->transfer_buffer, dev->packet_size );
	*(dev->read_queue+offset+(dev->packet_size)) = dev->serial_number++;

	atomic_set( &dev->intr_idx, aux_idx );
	spin_unlock( &dev->intr_idx_lock );

	wake_up_interruptible( &dev->read_wait );
}


/**
 *	iowarrior_debug_data
 */
static inline void iowarrior_debug_data (const char *function, int size, const unsigned char *data)
{
	int i;

	if (!debug)
		return;

	printk (KERN_DEBUG __FILE__": %s - length = %d, data = ",
		function, size);
	for (i = 0; i < size; ++i)
	{
		printk ("%.2x ", data[i]);
	}
	printk ("\n");
}
